/****************************************************************************
**
** Copyright (C) 2021 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the QtQml module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:COMM$
**
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** $QT_END_LICENSE$
**
**
**
**
**
**
**
**
**
**
**
**
**
**
**
**
**
**
**
****************************************************************************/

#ifndef QFIELDLIST_P_H
#define QFIELDLIST_P_H

//
//  W A R N I N G
//  -------------
//
// This file is not part of the Qt API.  It exists purely as an
// implementation detail.  This header file may change from version to
// version without notice, or even be removed.
//
// We mean it.
//

#include <QtCore/qglobal.h>

#include <private/qflagpointer_p.h>

// QForwardFieldList is a super simple linked list that can only prepend
template<class N, N *N::*nextMember>
class QForwardFieldList
{
public:
    inline QForwardFieldList();
    inline N *first() const;
    inline N *takeFirst();

    inline void prepend(N *);

    inline bool isEmpty() const;
    inline bool isOne() const;
    inline bool isMany() const;

    static inline N *next(N *v);

    inline bool flag() const;
    inline void setFlag();
    inline void clearFlag();
    inline void setFlagValue(bool);

    inline bool flag2() const;
    inline void setFlag2();
    inline void clearFlag2();
    inline void setFlag2Value(bool);
private:
    QFlagPointer<N> _first;
};

// QFieldList is a simple linked list, that can append and prepend and also
// maintains a count
template<class N, N *N::*nextMember>
class QFieldList
{
public:
    inline QFieldList();
    inline N *first() const;
    inline N *takeFirst();

    inline void append(N *);
    inline void prepend(N *);

    inline bool isEmpty() const;
    inline bool isOne() const;
    inline bool isMany() const;
    inline int count() const;

    inline void append(QFieldList<N, nextMember> &);
    inline void prepend(QFieldList<N, nextMember> &);
    inline void insertAfter(N *, QFieldList<N, nextMember> &);

    inline void copyAndClear(QFieldList<N, nextMember> &);
    inline void copyAndClearAppend(QForwardFieldList<N, nextMember> &);
    inline void copyAndClearPrepend(QForwardFieldList<N, nextMember> &);

    static inline N *next(N *v);

    inline bool flag() const;
    inline void setFlag();
    inline void clearFlag();
    inline void setFlagValue(bool);
private:
    N *_first;
    N *_last;
    quint32 _flag:1;
    quint32 _count:31;
};

template<class N, N *N::*nextMember>
QForwardFieldList<N, nextMember>::QForwardFieldList()
{
}

template<class N, N *N::*nextMember>
N *QForwardFieldList<N, nextMember>::first() const
{
    return *_first;
}

template<class N, N *N::*nextMember>
N *QForwardFieldList<N, nextMember>::takeFirst()
{
    N *value = *_first;
    if (value) {
        _first = next(value);
        value->*nextMember = nullptr;
    }
    return value;
}

template<class N, N *N::*nextMember>
void QForwardFieldList<N, nextMember>::prepend(N *v)
{
    Q_ASSERT(v->*nextMember == nullptr);
    v->*nextMember = *_first;
    _first = v;
}

template<class N, N *N::*nextMember>
bool QForwardFieldList<N, nextMember>::isEmpty() const
{
    return _first.isNull();
}

template<class N, N *N::*nextMember>
bool QForwardFieldList<N, nextMember>::isOne() const
{
    return *_first && _first->*nextMember == 0;
}

template<class N, N *N::*nextMember>
bool QForwardFieldList<N, nextMember>::isMany() const
{
    return *_first && _first->*nextMember != 0;
}

template<class N, N *N::*nextMember>
N *QForwardFieldList<N, nextMember>::next(N *v)
{
    Q_ASSERT(v);
    return v->*nextMember;
}

template<class N, N *N::*nextMember>
bool QForwardFieldList<N, nextMember>::flag() const
{
    return _first.flag();
}

template<class N, N *N::*nextMember>
void QForwardFieldList<N, nextMember>::setFlag()
{
    _first.setFlag();
}

template<class N, N *N::*nextMember>
void QForwardFieldList<N, nextMember>::clearFlag()
{
    _first.clearFlag();
}

template<class N, N *N::*nextMember>
void QForwardFieldList<N, nextMember>::setFlagValue(bool v)
{
    _first.setFlagValue(v);
}

template<class N, N *N::*nextMember>
bool QForwardFieldList<N, nextMember>::flag2() const
{
    return _first.flag2();
}

template<class N, N *N::*nextMember>
void QForwardFieldList<N, nextMember>::setFlag2()
{
    _first.setFlag2();
}

template<class N, N *N::*nextMember>
void QForwardFieldList<N, nextMember>::clearFlag2()
{
    _first.clearFlag2();
}

template<class N, N *N::*nextMember>
void QForwardFieldList<N, nextMember>::setFlag2Value(bool v)
{
    _first.setFlag2Value(v);
}

template<class N, N *N::*nextMember>
QFieldList<N, nextMember>::QFieldList()
: _first(nullptr), _last(nullptr), _flag(0), _count(0)
{
}

template<class N, N *N::*nextMember>
N *QFieldList<N, nextMember>::first() const
{
    return _first;
}

template<class N, N *N::*nextMember>
N *QFieldList<N, nextMember>::takeFirst()
{
    N *value = _first;
    if (value) {
        _first = next(value);
        if (_last == value) {
            Q_ASSERT(_first == nullptr);
            _last = nullptr;
        }
        value->*nextMember = nullptr;
        --_count;
    }
    return value;
}

template<class N, N *N::*nextMember>
void QFieldList<N, nextMember>::append(N *v)
{
    Q_ASSERT(v->*nextMember == nullptr);
    if (isEmpty()) {
        _first = v;
        _last = v;
    } else {
        _last->*nextMember = v;
        _last = v;
    }
    ++_count;
}

template<class N, N *N::*nextMember>
void QFieldList<N, nextMember>::prepend(N *v)
{
    Q_ASSERT(v->*nextMember == nullptr);
    if (isEmpty()) {
        _first = v;
        _last = v;
    } else {
        v->*nextMember = _first;
        _first = v;
    }
    ++_count;
}

template<class N, N *N::*nextMember>
bool QFieldList<N, nextMember>::isEmpty() const
{
    return _count == 0;
}

template<class N, N *N::*nextMember>
bool QFieldList<N, nextMember>::isOne() const
{
    return _count == 1;
}

template<class N, N *N::*nextMember>
bool QFieldList<N, nextMember>::isMany() const
{
    return _count > 1;
}

template<class N, N *N::*nextMember>
int QFieldList<N, nextMember>::count() const
{
    return _count;
}

template<class N, N *N::*nextMember>
N *QFieldList<N, nextMember>::next(N *v)
{
    Q_ASSERT(v);
    return v->*nextMember;
}

template<class N, N *N::*nextMember>
void QFieldList<N, nextMember>::append(QFieldList<N, nextMember> &o)
{
    if (!o.isEmpty()) {
        if (isEmpty()) {
            _first = o._first;
            _last = o._last;
            _count = o._count;
        } else {
            _last->*nextMember = o._first;
            _last = o._last;
            _count += o._count;
        }
        o._first = o._last = 0; o._count = 0;
    }
}

template<class N, N *N::*nextMember>
void QFieldList<N, nextMember>::prepend(QFieldList<N, nextMember> &o)
{
    if (!o.isEmpty()) {
        if (isEmpty()) {
            _first = o._first;
            _last = o._last;
            _count = o._count;
        } else {
            o._last->*nextMember = _first;
            _first = o._first;
            _count += o._count;
        }
        o._first = o._last = 0; o._count = 0;
    }
}

template<class N, N *N::*nextMember>
void QFieldList<N, nextMember>::insertAfter(N *after, QFieldList<N, nextMember> &o)
{
    if (after == 0) {
        prepend(o);
    } else if (after == _last) {
        append(o);
    } else if (!o.isEmpty()) {
        if (isEmpty()) {
            _first = o._first;
            _last = o._last;
            _count = o._count;
        } else {
            o._last->*nextMember = after->*nextMember;
            after->*nextMember = o._first;
            _count += o._count;
        }
        o._first = o._last = 0; o._count = 0;
    }
}

template<class N, N *N::*nextMember>
void QFieldList<N, nextMember>::copyAndClear(QFieldList<N, nextMember> &o)
{
    _first = o._first;
    _last = o._last;
    _count = o._count;
    o._first = o._last = nullptr;
    o._count = 0;
}

template<class N, N *N::*nextMember>
void QFieldList<N, nextMember>::copyAndClearAppend(QForwardFieldList<N, nextMember> &o)
{
    _first = 0;
    _last = 0;
    _count = 0;
    while (N *n = o.takeFirst()) append(n);
}

template<class N, N *N::*nextMember>
void QFieldList<N, nextMember>::copyAndClearPrepend(QForwardFieldList<N, nextMember> &o)
{
    _first = nullptr;
    _last = nullptr;
    _count = 0;
    while (N *n = o.takeFirst()) prepend(n);
}

template<class N, N *N::*nextMember>
bool QFieldList<N, nextMember>::flag() const
{
    return _flag;
}

template<class N, N *N::*nextMember>
void QFieldList<N, nextMember>::setFlag()
{
    _flag = true;
}

template<class N, N *N::*nextMember>
void QFieldList<N, nextMember>::clearFlag()
{
    _flag = false;
}

template<class N, N *N::*nextMember>
void QFieldList<N, nextMember>::setFlagValue(bool v)
{
    _flag = v;
}

#endif // QFIELDLIST_P_H
